/*
 * Decompiled with CFR 0.152.
 */
package net.pcal.fastback.tasks;

import com.google.common.collect.ListMultimap;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import net.pcal.fastback.ModContext;
import net.pcal.fastback.WorldConfig;
import net.pcal.fastback.logging.Logger;
import net.pcal.fastback.logging.Message;
import net.pcal.fastback.progress.IncrementalProgressMonitor;
import net.pcal.fastback.progress.PercentageProgressMonitor;
import net.pcal.fastback.utils.GitUtils;
import net.pcal.fastback.utils.SnapshotId;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.merge.ContentMergeStrategy;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.TrackingRefUpdate;
import org.eclipse.jgit.transport.URIish;

public class PushTask
implements Callable<Void> {
    private final ModContext ctx;
    private final Logger log;
    private final Git git;
    private final SnapshotId sid;

    public PushTask(Git git, ModContext ctx, Logger log, SnapshotId sid) {
        this.git = Objects.requireNonNull(git);
        this.ctx = Objects.requireNonNull(ctx);
        this.log = Objects.requireNonNull(log);
        this.sid = Objects.requireNonNull(sid);
    }

    @Override
    public Void call() throws GitAPIException, IOException {
        this.log.hud(Message.localized("fastback.hud.remote-uploading", 0));
        WorldConfig worldConfig = WorldConfig.load(this.git);
        String pushUrl = worldConfig.getRemotePushUrl();
        if (pushUrl == null) {
            String msg = "Skipping remote backup because no remote url has been configured.";
            this.log.warn("Skipping remote backup because no remote url has been configured.");
            return null;
        }
        Collection remoteBranchRefs = this.git.lsRemote().setHeads(true).setTags(false).setRemote(worldConfig.getRemoteName()).call();
        ListMultimap<String, SnapshotId> snapshotsPerWorld = SnapshotId.getSnapshotsPerWorld(remoteBranchRefs, this.log);
        if (worldConfig.isUuidCheckEnabled()) {
            boolean uuidCheckResult;
            try {
                uuidCheckResult = PushTask.doUuidCheck(this.git, snapshotsPerWorld.keySet(), worldConfig, this.log);
            }
            catch (IOException | GitAPIException e) {
                this.log.internalError("Skipping remote backup due to failed uuid check", e);
                return null;
            }
            if (!uuidCheckResult) {
                this.log.warn("Skipping remote backup due to world mismatch.");
                return null;
            }
        }
        this.log.info("Pushing to " + worldConfig.getRemotePushUrl());
        if (worldConfig.isSmartPushEnabled()) {
            PushTask.doSmartPush(this.git, snapshotsPerWorld.get((Object)worldConfig.worldUuid()), this.sid.getBranchName(), worldConfig, this.log);
        } else {
            PushTask.doNaivePush(this.git, this.sid.getBranchName(), worldConfig, this.log);
        }
        this.log.info("Remote backup complete.");
        return null;
    }

    private static void doSmartPush(Git git, List<SnapshotId> remoteSnapshots, String branchNameToPush, WorldConfig worldConfig, Logger logger) throws GitAPIException, IOException {
        String remoteName = worldConfig.getRemoteName();
        String worldUuid = worldConfig.worldUuid();
        if (remoteSnapshots.isEmpty()) {
            logger.warn("** This appears to be the first time this world has been pushed.");
            logger.warn("** If the world is large, this may take some time.");
            PushTask.doNaivePush(git, branchNameToPush, worldConfig, logger);
            return;
        }
        List localBranchRefs = git.branchList().call();
        ListMultimap<String, SnapshotId> localSnapshotsPerWorld = SnapshotId.getSnapshotsPerWorld(localBranchRefs, logger);
        List localSnapshots = localSnapshotsPerWorld.get((Object)worldUuid);
        remoteSnapshots.retainAll(localSnapshots);
        if (remoteSnapshots.isEmpty()) {
            logger.warn("No common snapshots found between local and remote.");
            logger.warn("Doing a full push.  This may take some time.");
            PushTask.doNaivePush(git, branchNameToPush, worldConfig, logger);
            return;
        }
        Collections.sort(remoteSnapshots);
        SnapshotId latestCommonSnapshot = remoteSnapshots.get(remoteSnapshots.size() - 1);
        logger.info("Using existing snapshot " + latestCommonSnapshot + " for common history");
        String tempBranchName = PushTask.getTempBranchName(branchNameToPush);
        logger.debug("Creating out temp branch " + tempBranchName);
        git.checkout().setCreateBranch(true).setName(tempBranchName).call();
        ObjectId branchId = git.getRepository().resolve(latestCommonSnapshot.getBranchName());
        logger.debug("Merging " + latestCommonSnapshot.getBranchName());
        git.merge().setContentMergeStrategy(ContentMergeStrategy.OURS).include((AnyObjectId)branchId).setMessage("Merge " + branchId + " into " + tempBranchName).call();
        logger.debug("Checking out " + branchNameToPush);
        git.checkout().setName(branchNameToPush).call();
        logger.debug("Pushing temp branch " + tempBranchName);
        IncrementalProgressMonitor pm = new IncrementalProgressMonitor(new PushProgressMonitor(logger), 100);
        Iterable pushResult = git.push().setProgressMonitor((ProgressMonitor)pm).setRemote(remoteName).setRefSpecs(new RefSpec[]{new RefSpec(tempBranchName + ":" + tempBranchName), new RefSpec(branchNameToPush + ":" + branchNameToPush)}).call();
        logger.info("Cleaning up branches...");
        if (worldConfig.isTrackingBranchCleanupEnabled()) {
            for (PushResult pr : pushResult) {
                for (TrackingRefUpdate f : pr.getTrackingRefUpdates()) {
                    String PREFIX = "refs/remotes/";
                    if (f.getLocalName().startsWith("refs/remotes/")) {
                        String trackingBranchName = f.getLocalName().substring("refs/remotes/".length());
                        logger.info("Cleaning up tracking branch " + trackingBranchName);
                        git.branchDelete().setForce(true).setBranchNames(new String[]{trackingBranchName}).call();
                        continue;
                    }
                    logger.warn("Ignoring unrecognized TrackingRefUpdate " + f.getLocalName());
                }
            }
        }
        if (worldConfig.isTempBranchCleanupEnabled()) {
            logger.info("Deleting local temp branch " + tempBranchName);
            git.branchDelete().setForce(true).setBranchNames(new String[]{tempBranchName}).call();
        }
        if (worldConfig.isRemoteTempBranchCleanupEnabled()) {
            String remoteTempBranch = "refs/heads/" + tempBranchName;
            logger.info("Deleting remote temp branch " + remoteTempBranch);
            RefSpec deleteRemoteBranchSpec = new RefSpec().setSource(null).setDestination(remoteTempBranch);
            git.push().setProgressMonitor((ProgressMonitor)pm).setRemote(remoteName).setRefSpecs(new RefSpec[]{deleteRemoteBranchSpec}).call();
        }
        logger.info("Push complete");
    }

    private static void doNaivePush(Git git, String branchNameToPush, WorldConfig config, Logger logger) throws IOException, GitAPIException {
        IncrementalProgressMonitor pm = new IncrementalProgressMonitor(new PushProgressMonitor(logger), 100);
        String remoteName = config.getRemoteName();
        logger.info("Doing naive push of " + branchNameToPush);
        git.push().setProgressMonitor((ProgressMonitor)pm).setRemote(remoteName).setRefSpecs(new RefSpec[]{new RefSpec(branchNameToPush + ":" + branchNameToPush)}).call();
    }

    private static boolean doUuidCheck(Git git, Set<String> remoteWorldUuids, WorldConfig config, Logger logger) throws GitAPIException, IOException {
        String localUuid = config.worldUuid();
        if (remoteWorldUuids.size() > 2) {
            logger.warn("Remote has more than one world-uuid.  This is unusual. " + remoteWorldUuids);
        }
        if (remoteWorldUuids.isEmpty()) {
            logger.debug("Remote does not have any previously-backed up worlds.");
        } else if (!remoteWorldUuids.contains(localUuid)) {
            URIish remoteUri = GitUtils.getRemoteUri(git, config.getRemoteName(), logger);
            logger.chatError(Message.localized("fastback.chat.push-uuid-mismatch", remoteUri));
            logger.info("local: " + localUuid + ", remote: " + remoteWorldUuids);
            return false;
        }
        logger.debug("world-uuid check passed.");
        return true;
    }

    static boolean isTempBranch(String branchName) {
        return branchName.startsWith("temp/");
    }

    private static String getTempBranchName(String uniqueName) {
        return "temp/" + uniqueName;
    }

    private static class PushProgressMonitor
    extends PercentageProgressMonitor {
        private final Logger logger;

        public PushProgressMonitor(Logger logger) {
            this.logger = Objects.requireNonNull(logger);
        }

        @Override
        public void progressStart(String task) {
            this.logger.info(task);
        }

        @Override
        public void progressUpdate(String task, int percentage) {
            this.logger.info(task + " " + percentage + "%");
            if (task.contains("Finding sources")) {
                this.logger.hud(Message.localized("fastback.hud.remote-preparing", percentage / 2));
            } else if (task.contains("Writing objects")) {
                this.logger.hud(Message.localized("fastback.hud.remote-uploading", 50 + percentage / 2));
            }
        }

        @Override
        public void progressDone(String task) {
            this.logger.info("Done " + task);
        }

        public void showDuration(boolean enabled) {
        }
    }
}

